<?php

namespace App\Http\Controllers\Api\V1\User;

use App\Constants\GlobalConst;
use App\Constants\NotificationConst;
use App\Constants\PaymentGatewayConst;
use App\Http\Controllers\Controller;
use App\Http\Helpers\CardyFieHelper;
use App\Http\Helpers\NotificationHelper;
use App\Http\Helpers\Response;
use App\Models\Admin\BasicSettings;
use App\Models\CardyfieCardCustomer;
use App\Models\CardyfieVirtualCard;
use App\Models\Transaction;
use App\Models\UserNotification;
use App\Models\UserWallet;
use App\Models\VirtualCardApi;
use App\Notifications\Admin\ActivityNotification;
use App\Notifications\User\VirtualCard\CreateMail;
use App\Notifications\User\VirtualCard\Fund;
use App\Notifications\User\VirtualCard\Withdraw;
use App\Providers\Admin\BasicSettingsProvider;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class CardyFieVirtualCardController extends Controller
{
    protected $api;
    protected $card_limit;
    protected $basic_settings;
    protected $card_supported_currencies;
    protected $cardCharges;


    public function __construct(){
        $cardApi = VirtualCardApi::first();
        $this->api =  $cardApi;
        $this->card_limit =  $cardApi->card_limit;
        $this->basic_settings = BasicSettingsProvider::get();
    }

    public function index()
    {
        $user = auth()->user();
        $card_basic_info = [
            'card_create_limit' => $this->api->card_limit,
            'card_back_details' => $this->api->card_details,
            'card_bg'           => get_image(@$this->api->image,'card-api'),
            'site_title'        => $this->basic_settings->site_name,
            'site_logo'         => get_logo(@$this->basic_settings,'dark'),
            'site_fav'          => get_fav($this->basic_settings,'dark'),
        ];


        if($this->api->config->mode == GlobalConst::SANDBOX){
            $myCards = CardyfieVirtualCard::where('user_id',$user->id)->where('env',strtoupper(GlobalConst::SANDBOX))->latest()->limit($this->card_limit)->get();
        }
        else{
            $myCards = CardyfieVirtualCard::where('user_id',$user->id)->whereNot('env',strtoupper(GlobalConst::SANDBOX))->latest()->limit($this->card_limit)->get();
        }

        $myCards = $myCards->map(function($data){
            $statusInfo = [
                     "PROCESSING" => "PROCESSING",
                     "ENABLED"    => "ENABLED",
                     "FREEZE"     => "FREEZE",
                     "CLOSED"     => "CLOSED",
                ];
            return[
                'id'            => $data->id,
                'reference_id'  => $data->reference_id,
                'ulid'          => $data->ulid,
                'customer_ulid' => $data->customer_ulid,
                'card_name'     => $data->card_name,
                'amount'        => floatval($data->amount ?? 0),
                'currency'      => $data->currency,
                'card_tier'     => $data->card_tier ?? '',
                'card_type'     => $data->card_type ?? '',
                'card_exp_time' => $data->card_exp_time,
                'masked_pan'    => $data->masked_pan,
                'address'       => $data->address,
                'status'        => $data->status,
                'env'           => $data->env,
                'is_default'    => $data->is_default,
                'status_info'   => (object)$statusInfo,
            ];
        });

        $transactions = Transaction::auth()->virtualCard()->latest()->get()->map(function($item){
            $statusInfo = [
                "success" =>      1,
                "pending" =>      2,
                "rejected" =>     3,
            ];
            $card_currency = $item->details->card_info->currency ?? get_default_currency_code();

            return[
                'id'                    => $item->id,
                'trx'                   => $item->trx_id,
                'transaction_type'      => "Virtual Card".'('. @$item->remark.')',
                'request_amount'        => getAmount($item->request_amount, get_wallet_precision()).' '.$card_currency ,
                'payable'               => getAmount($item->payable, get_wallet_precision()).' '.get_default_currency_code(),
                'total_charge'          => getAmount($item->total_charge, get_wallet_precision()).' '.get_default_currency_code(),
                'card_amount'           => getAmount(@$item->details->card_info->amount, get_wallet_precision()).' '.$card_currency,
                'card_number'           => $item->details->card_info->card_pan??$item->details->card_info->maskedPan??$item->details->card_info->card_number??"",
                'current_balance'       => getAmount($item->available_balance, get_wallet_precision()).' '.get_default_currency_code(),
                'status'                => $item->stringStatus->value,
                'date_time'             => $item->created_at,
                'status_info'           =>(object)$statusInfo,

            ];
        });
        $userWallet = auth()->guard('api')->user()->wallets()->with("currency:id,code,rate,flag,symbol,type,default,name")->get();


        $card_customer = $user->cardyfieCardCustomer()->where('env',$this->api->config->mode)->first();
        $totalCards = CardyfieVirtualCard::where('user_id',$user->id)->where('env',$this->api->config->mode)->count() ?? 0;

        $supported_currency = $this->api->currencies;
        $get_remaining_fields = [
            'transaction_type'  =>  PaymentGatewayConst::VIRTUALCARD,
            'attribute'         =>  PaymentGatewayConst::RECEIVED,
        ];
        $data =[
            'base_curr'                 => get_default_currency_code(),
            'base_curr_rate'            => getAmount(get_default_currency_rate(),get_wallet_precision()),
            'get_remaining_fields'      => (object) $get_remaining_fields,
            'supported_currency'        => $supported_currency,
            'card_create_action'        => $totalCards <  $this->card_limit ? true : false,
            'customer_create_status'    => $card_customer == null ? true : false,
            'card_basic_info'           => (object) $card_basic_info,
            'my_cards'                  => $myCards,
            'user'                      => $user,
            'user_wallet'               => (object)$userWallet,
            // 'card_charge'            => (object)$cardyfie_card_charge,
            'transactions'              => $transactions,
        ];

        return Response::success([__('Virtual Card Info')],$data);
    }


    public function createPage(){
        $user       = auth()->user();
        $customer_exist_status  = $user->cardyfieCardCustomer()->where('env',$this->api->config->mode)->first() != null ? true : false;
        $card_create_status     = ($user->cardyfieCardCustomer()->where('env',$this->api->config->mode)->first()?->status == GlobalConst::CARD_CUSTOMER_APPROVED_STATUS)  ? true : false;


        if( $customer_exist_status){
            $customer_exist =  $user->cardyfieCardCustomer()->where('env',$this->api->config->mode)->first();
        }else{
            $customer_exist =[
                "id"             => "",
                "user_type"      => "",
                "user_id"        => "",
                "ulid"           => "",
                "reference_id"   => "",
                "first_name"     => "",
                "last_name"      => "",
                "email"          => "",
                "date_of_birth"  => "",
                "id_number"      => "",
                "id_front_image" => "",
                "id_back_image"  => "",
                "user_image"     => "",
                "house_number"   => "",
                "address_line_1" => "",
                "city"           => "",
                "state"          => "",
                "zip_code"       => "",
                "country"        => "",
                "status"         => "",
                "meta"           => "",
                "state"          => "",
                "created_at"     => "",
                "updated_at"     => "",
                "idFontImage"    => "",
                "idBackImage"    => "",
                "userImage"      => "",
            ];
            $customer_exist = (object) $customer_exist;
        }
       $customer_pending_text = __("Please wait until your customer status is APPROVED. Once it is APPROVED, you can continue with the card creation.");



        $data =[
            'user'                  => $user,
            'customer_exist_status' => $customer_exist_status,
            'customer_pending_text' => $customer_pending_text,
            'customer_exist'        => $customer_exist,
            'card_create_status'    => $card_create_status,
        ];

        return Response::success([__('Virtual Card Info')],$data);

    }


    public function createCustomer(Request $request){

        $validator = Validator::make($request->all(), [
            'first_name'      => 'required|string|max:100',
            'last_name'       => 'required|string|max:100',
            'email'           => 'required|email|max:150|unique:cardyfie_card_customers,email', // check unique in users table
            'date_of_birth'   => 'required|date',
            'identity_type'   => 'required|in:nid,passport,bvn',
            'identity_number' => 'required|string|max:50',
            'id_front_image'  => 'required|image|mimes:jpg,jpeg,png,svg,webp|max:10240',
            'id_back_image'   => 'required|image|mimes:jpg,jpeg,png,svg,webp|max:10240',
            'user_image'      => 'required|image|mimes:jpg,jpeg,png,svg,webp|max:10240',
            'house_number'    => 'required|string|max:150',
            'country'         => 'required|string|max:100',
            'city'            => 'required|string|max:100',
            'state'           => 'required|string|max:100',
            'zip_code'        => 'required|string|max:20',
            'address'         => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return Response::validation($validator->errors()->all(),null);
        }

        $validated  = $validator->validate();
        $user       = auth()->user();

        $card_customer = $user->cardyfieCardCustomer()->where('env',$this->api->config->mode)->first();
        try{
            if( $card_customer == null){
                if ($request->hasFile("id_front_image")) {
                    $imageData = saveImageAndGetUrl($request->file("id_front_image"), 'card-kyc-images');
                    $validated['id_front_image'] = $imageData['url'];
                    $uploadedImages[] = $imageData['path'];
                }

                // id back image
                if ($request->hasFile("id_back_image")) {
                    $imageData = saveImageAndGetUrl($request->file("id_back_image"), 'card-kyc-images');
                    $validated['id_back_image'] = $imageData['url'];
                    $uploadedImages[] = $imageData['path'];
                }

                // user image
                if ($request->hasFile("user_image")) {
                    $imageData = saveImageAndGetUrl($request->file("user_image"), 'card-kyc-images');
                    $validated['user_image'] = $imageData['url'];
                    $uploadedImages[] = $imageData['path'];
                }
                $validated['reference_id'] = "ref-".getTrxNum();

                // Clean mapping for DB insert
                    $storeData =  CardyfieCardCustomer::create([
                        'user_type'      => "USER",
                        'user_id'        => $user->id,
                        'reference_id'   => $validated['reference_id'],
                        'first_name'     => $validated['first_name'],
                        'last_name'      => $validated['last_name'],
                        'email'          => $validated['email'],
                        'date_of_birth'  => $validated['date_of_birth'],
                        'id_type'        => $validated['identity_type'],
                        'id_number'      => $validated['identity_number'],
                        'id_front_image' => $validated['id_front_image'],
                        'id_back_image'  => $validated['id_back_image'],
                        'user_image'     => $validated['user_image'],
                        'house_number'   => $validated['house_number'],
                        'address_line_1' => $validated['address'],
                        'city'           => $validated['city'],
                        'state'          => $validated['state'],
                        'zip_code'       => $validated['zip_code'],
                        'country'        => $validated['country'],
                        'status'         => "PENDING",
                        'env'            => $this->api->config->mode
                    ]);



                $createCustomer = (new CardyFieHelper())->createCustomer($validated);

                if( $createCustomer['status'] == false){
                    // delete all uploaded images if API failed
                    foreach ($uploadedImages as $filePath) {
                        if (file_exists($filePath)) {
                            unlink($filePath);
                        }
                    }
                    $storeData->delete();
                    return $this->apiErrorHandle($createCustomer["message"]);

                }

                try{
                    if($createCustomer['message'][0] == "Customer already exists with this email address!"){
                        foreach ($uploadedImages as $filePath) {
                            if (file_exists($filePath)) {
                                unlink($filePath);
                            }
                        }
                        $storeData->delete();

                        return Response::success([$createCustomer['message'][0]],null);

                    }

                }catch(Exception $e){
                    return Response::error([ __("Something went wrong! Please try again.")],null);
                }

                //update few fields after store
                $storeData->ulid   = $createCustomer['data']['customer']['ulid'];
                $storeData->meta   = $createCustomer['data']['customer']['meta'];
                $storeData->status = $createCustomer['data']['customer']['status'];
                $storeData->save();
            }

            return Response::success([$createCustomer['message'][0] ?? __('Customer has been created successfully.')],null);

        }catch(Exception $e){
            return Response::error([ __("Something went wrong! Please try again.")],null);
        }
    }

    public function editCustomerPage(){
        $user          = auth()->user();
        $card_customer = $user->cardyfieCardCustomer()->where('env',$this->api->config->mode)->first();

        if(!$card_customer){;
            Response::error([ __("Something went wrong! Please try again.")],null);
        }

        //check customer have on cardyfie platform or not
        $getCustomer = (new CardyFieHelper())->getCustomer( (string)$card_customer->ulid);

        if( $getCustomer['status'] == false){
            return $this->apiErrorHandle($getCustomer["message"]);
        }
        $data =[
            'card_customer' => $card_customer,
        ];

        return Response::success([__('Data Fetch Successful')],$data);

    }


    public function updateCustomer(Request $request){

        $validator = Validator::make($request->all(), [
            'first_name'      => 'required|string|max:100',
            'last_name'       => 'required|string|max:100',
            'date_of_birth'   => 'required|date',
            'identity_type'   => 'required|in:nid,passport,bvn',
            'identity_number' => 'required|string|max:50',
            'id_front_image'  => 'nullable|image|mimes:jpg,jpeg,png,svg,webp|max:10240',
            'id_back_image'   => 'nullable|image|mimes:jpg,jpeg,png,svg,webp|max:10240',
            'user_image'      => 'nullable|image|mimes:jpg,jpeg,png,svg,webp|max:10240',
            'house_number'    => 'required|string|max:150',
            'city'            => 'required|string|max:100',
            'state'           => 'required|string|max:100',
            'zip_code'        => 'required|string|max:20',
            'address'         => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return Response::validation($validator->errors()->all(),null);

        }

        $validated  = $validator->validate();

        $user       = auth()->user();
        $card_customer = $user->cardyfieCardCustomer()->where('env',$this->api->config->mode)->first();
        if(!$card_customer){
            return Response::error([ __("Something went wrong! Please try again.")],null);
        }

        try{

            if ($request->hasFile("id_front_image")) {
                $imageData = saveImageAndGetUrl($request->file("id_front_image"), 'card-kyc-images');
                $validated['id_front_image'] = $imageData['url'];
                $uploadedImages[] = $imageData['path'];
            }

            // id back image
            if ($request->hasFile("id_back_image")) {
                $imageData = saveImageAndGetUrl($request->file("id_back_image"), 'card-kyc-images');
                $validated['id_back_image'] = $imageData['url'];
                $uploadedImages[] = $imageData['path'];
            }

            // user image
            if ($request->hasFile("user_image")) {
                $imageData = saveImageAndGetUrl($request->file("user_image"), 'card-kyc-images');
                $validated['user_image'] = $imageData['url'];
                $uploadedImages[] = $imageData['path'];
            }


            $updateCustomer = (new CardyFieHelper())->updateCustomer($validated,(string)$card_customer->ulid);

            if( $updateCustomer['status'] == false){
                // delete all uploaded images if API failed
                foreach ($uploadedImages as $filePath) {
                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }
                }
                return $this->apiErrorHandle($updateCustomer["message"]);

            }
            try{
                if ($card_customer) {
                    $customerData = $updateCustomer['data']['customer'];

                    $card_customer->update(array_merge([
                        'user_type' => 'USER',
                        'user_id'   => $user->id,
                    ], $customerData));
                }

            }catch(Exception $e){
                return Response::error([ __("Something went wrong! Please try again.")],null);
            }

            return Response::success([__('Customer has been updated successfully.')],null);

        }catch(Exception $e){
            return Response::error([ __("Something went wrong! Please try again.")],null);
        }

    }

    public function cardBuy(Request $request){

        $validator = Validator::make($request->all(), [
            'name_on_card'  => 'required|string|min:4|max:50',
            'card_tier'     => 'required|string|max:30',
            'card_type'     => 'required|string|max:30',
            'currency'      => "required|string",
            'from_currency' => "required|string|exists:currencies,code",
        ]);

        if ($validator->fails()) {
            return Response::validation($validator->errors()->all(),null);
        }

        $validated  = $validator->validate();

        $user          = auth()->user();
        $basic_setting = BasicSettings::first();


        $wallet = UserWallet::where('user_id',$user->id)->whereHas("currency",function($q) use ($validated) {
            $q->where("code",$validated['from_currency'])->active();
        })->active()->first();

        if(!$wallet){
            return Response::error([ __('User wallet not found')],null);
        }

        $card_currency = $this->api->currencies->first();
        $card_charge = $this->api->currencies->first()->fees;
        if(!$card_currency){
            return Response::error([ __('Card Currency Not Found')],null);
        }

        // $cardCharge =  $this->cardCharges;
        $charges = $this->cardIssuesCharge($validated['card_tier'],$card_charge,$wallet,$card_currency);


        if($charges['payable'] > $wallet->balance){
            return Response::error([ __("Your Wallet Balance Is Insufficient")],null);
        }

        $customer = $user->cardyfieCardCustomer;
        if(!$customer){
            return Response::error([ __("The customer doesn't create properly,Contact with owner")],null);
        }

        $customer_card  = CardyfieVirtualCard::where('user_id',$user->id)->count() ?? 0;
        if($customer_card >= $this->card_limit){
            Response::error([__("Sorry! You can not create more than")." ".$this->card_limit ." ".__("card using the same email address.")],null);

        }
        $validated['customer_ulid'] = $customer->ulid;
        $validated['reference_id'] = "ref-".getTrxNum();

        DB::beginTransaction();
        try{
            //store card info to db
            $card_id = DB::table("cardyfie_virtual_cards")->insertGetId([
                'user_type'     => "USER",
                'user_id'       => $user->id,
                'reference_id'  => $validated['reference_id'],
                'customer_ulid' => $validated['customer_ulid'],
                'card_name'     => $validated['name_on_card'],
                'amount'        => floatVal(0),
                'currency'      => $validated['currency'],
                'card_tier'     => $validated['card_tier'],
                'card_type'     => $validated['card_type'],
                'created_at'    => now(),
                'updated_at'    => now(),
            ]);

            //create card from cardyfie
            $createCard = (new CardyFieHelper())->issuesCard($validated);
            if( $createCard['status'] == false){
                return $this->apiErrorHandle($createCard["message"]);
            }


            //update card after success response;
            $card                = CardyfieVirtualCard::where('id',$card_id)->first();
            $card->ulid          = $createCard['data']['virtual_card']['ulid'] ?? null;
            $card->amount        = $createCard['data']['virtual_card']['card_balance'] ?? floatval(0);
            $card->card_exp_time = $createCard['data']['virtual_card']['card_exp_time'] ?? null;
            $card->masked_pan    = $createCard['data']['virtual_card']['masked_pan'] ?? null;
            $card->address       = $createCard['data']['virtual_card']['address'] ?? null;
            $card->status        = $createCard['data']['virtual_card']['status'] ?? null;
            $card->env           = $createCard['data']['virtual_card']['env'] ?? null;
            $card->save();


            $trx_id =  'CB'.getTrxNum();
            $sender = $this->insertCardBuy($trx_id,$user,$wallet,$card,$charges);
            $this->insertBuyCardCharge($charges,$user,$sender,$card->masked_pan);

            if($basic_setting->email_notification == true){
                $notifyDataSender = [
                        'trx_id'         => $trx_id,
                        'title'          => "Virtual Card (Buy Card)",
                        'request_amount' => get_amount($card->amount,$charges['card_currency'],$charges['precision_digit']),
                        'payable'        => get_amount($charges['payable'],$charges['from_currency'],$charges['precision_digit']),
                        'charges'        => get_amount( $charges['total_charge'],$charges['from_currency'],$charges['precision_digit']),
                        'card_amount'    => get_amount($card->amount,$charges['card_currency'],$charges['precision_digit']),
                        'card_pan'       => $card->masked_pan ?? "---- ----- ---- ----",
                        'status'         => $strowallet_card->card_status??"",
                    ];
                try{
                    $user->notify(new CreateMail($user,(object)$notifyDataSender));
                }catch(Exception $e){}
            }
            //admin notification
            $this->adminNotification($trx_id,$charges,$user,$card);


            DB::commit();

            return Response::success([__('Virtual Card Buy Successfully')],null);

        }catch(Exception $e){
            DB::rollBack();
            return Response::error([__("Something went wrong! Please try again.")],null);
        }

    }

     public function cardDeposit(Request $request){
        $validator = Validator::make($request->all(), [
            'card_id'        => 'required',
            'deposit_amount' => 'required|numeric|gt:0',
            'currency'       => "required|string",
            'from_currency'  => "required|string|exists:currencies,code",
        ]);

        if ($validator->fails()) {
            return Response::validation($validator->errors()->all(),null);
        }

        $validated     = $validator->validate();
        $basic_setting = BasicSettings::first();
        $user          = auth()->user();
        $amount        = $validated['deposit_amount'];

        $myCard =  CardyfieVirtualCard::where('user_id',$user->id)->where('ulid',$request->card_id)->first();
        if(!$myCard){
            return Response::error([__("Something is wrong in your card")],null);
        }

        $wallet = UserWallet::where('user_id',$user->id)->whereHas("currency",function($q) use ($validated) {
            $q->where("code",$validated['from_currency'])->active();
        })->active()->first();
        if(!$wallet){
            return Response::error([__('User wallet not found')],null);
        }

        $card_currency = $this->api->currencies->first();
        $card_charge = $this->api->currencies->first()->fees;
        $cardCharge = $card_currency;

        if(!$card_currency){
            return Response::error([ __('Card Currency Not Found')],null);
        }

        $charges = $this->cardDepositCharges($amount,$cardCharge,$wallet,$card_currency);

        $minLimit =  $cardCharge->min_limit;
        $maxLimit =  $cardCharge->max_limit;
        if($amount < $minLimit || $amount > $maxLimit){
            return Response::error([__("Please follow the transaction limit")],null);
        }

        if($charges['payable'] > $wallet->balance){
            return Response::error([ __("Your Wallet Balance Is Insufficient")],null);
        }

        $deposit_request  = (new CardyFieHelper())->depositCard( (string) $myCard->ulid, $amount);

        if( $deposit_request['status'] == false){
            return $this->apiErrorHandle($deposit_request["message"]);
        }


        try{
            //after success
            $myCard->amount = $deposit_request['data']['virtual_card']['card_balance'];
            $myCard->save();

            //create local transaction
            $trx_id = 'CF'.getTrxNum();
            $sender = $this->insertCardFund($trx_id,$user,$wallet,$amount,$myCard,$charges);
            $this->insertFundCardCharge($charges,$user,$sender,$myCard->masked_pan,$amount);
            if($basic_setting->email_notification == true){
                $notifyDataSender = [
                    'trx_id'        => $trx_id,
                    'request_amount'=> get_amount($amount,$charges['card_currency'],$charges['precision_digit']),
                    'payable'       => get_amount($charges['payable'],$charges['from_currency'],$charges['precision_digit']),
                    'charges'       => get_amount( $charges['total_charge'],$charges['from_currency'],$charges['precision_digit']),
                    'card_amount'   => get_amount($amount,$charges['card_currency'],$charges['precision_digit']),
                    'card_pan'      => $myCard->masked_pan ?? "---- ----- ---- ----",
                    'status'        => "Success",
                ];
                try{
                    $user->notify(new Fund($user,(object)$notifyDataSender));
                }catch(Exception $e){}
            }

            //admin notification
            $this->adminNotificationFund($trx_id,$charges,$amount,$user,$myCard);

            return Response::success([__('Card Funded Successfully')],null);
        }catch(Exception $e){
            return Response::error([__("Something went wrong! Please try again.")],null);
        }
    }


    public function cardWithdraw(Request $request){

        $validator = Validator::make($request->all(), [
            'card_id'           => 'required',
            'withdraw_amount'   => 'required|numeric|gt:0',
            'currency'          => "required|string",
            'from_currency'     => "required|string|exists:currencies,code",
        ]);

        if ($validator->fails()) {
            return Response::validation($validator->errors()->all(),null);
        }

        $validated     = $validator->validate();

        $basic_setting = BasicSettings::first();
        $user          = auth()->user();
        $amount        = $validated['withdraw_amount'];

        $myCard =  CardyfieVirtualCard::where('user_id',$user->id)->where('ulid',$request->card_id)->first();
        if(!$myCard){
            return Response::error([__("Something is wrong in your card")],null);
        }

        $get_card  = (new CardyFieHelper())->getCard( (string)$myCard->ulid);
        if( $get_card['status'] == false){
            return $this->apiErrorHandle($get_card["message"]);
        }

        //check real card balance
        if($amount > $get_card['data']['virtual_card']['card_balance']){
            return Response::error([__("Your card doesn't have enough funds to complete this transaction")],null);
        }

        $wallet = UserWallet::where('user_id',$user->id)->whereHas("currency",function($q) use ($validated) {
            $q->where("code",$validated['from_currency'])->active();
        })->active()->first();
        if(!$wallet){
            return Response::error([__('User wallet not found')],null);
        }

        $card_currency = $this->api->currencies->first();
        $card_charge = $this->api->currencies->first()->fees;
        $cardCharge = $card_currency;

        if(!$card_currency){
            return Response::error([ __('Card Currency Not Found')],null);
        }

        $charges = $this->cardWithdrawCharges($amount,$cardCharge,$wallet,$card_currency);

        $minLimit =  $cardCharge->min_limit;
        $maxLimit =  $cardCharge->max_limit;
        if($amount < $minLimit || $amount > $maxLimit){
            return Response::error([__("Please follow the transaction limit")],null);
        }

        $withdraw_request  = (new CardyFieHelper())->withdrawCard( (string) $myCard->ulid, $amount);
        if( $withdraw_request['status'] == false){
            return $this->apiErrorHandle($withdraw_request["message"]);
        }

        try{
            //after success
            $myCard->amount -= $amount;
            $myCard->save();


            //create local transaction
            $trx_id = 'CW-'.getTrxNum();
            $sender = $this->insertCardWithdrawal($trx_id,$user,$wallet,$amount,$myCard,$charges);
            $this->insertWithdrawalCardCharge($charges,$user,$sender,$myCard->masked_pan,$amount);
            if($basic_setting->email_notification == true){
                $notifyDataSender = [
                    'trx_id'        => $trx_id,
                    'request_amount'=> get_amount($amount,$charges['card_currency'],$charges['precision_digit']),
                    'payable'       => get_amount($charges['payable'],$charges['from_currency'],$charges['precision_digit']),
                    'charges'       => get_amount( $charges['total_charge'],$charges['from_currency'],$charges['precision_digit']),
                    'card_amount'   => get_amount($amount,$charges['card_currency'],$charges['precision_digit']),
                    'card_pan'      => $myCard->masked_pan ?? "---- ----- ---- ----",
                    'status'        => "Success",
                ];
                try{
                    $user->notify(new Withdraw($user,(object)$notifyDataSender));
                }catch(Exception $e){}
            }

            //admin notification
            $this->adminNotificationWithdraw($trx_id,$charges,$amount,$user,$myCard);

             return Response::success([__('Card Withdrawal Successful')],null);
        }catch(Exception $e){
            return Response::error([__("Something went wrong! Please try again.")],null);
        }

    }


    public function cardDetails(){
        $validator = Validator::make(request()->all(), [
            'card_id'     => "required|string",
        ]);
        if($validator->fails()){
            return Response::validation($validator->errors()->all(),null);
        }

        $validated  = $validator->validate();
        $user = auth()->user();
        $myCard = CardyfieVirtualCard::where('user_id',$user->id)->where('ulid',$validated['card_id'])->first();
        if(!$myCard){
            return Response::error([__("Something is wrong in your card")],null);
        }

        $get_card  = (new CardyFieHelper())->getCard( (string)$myCard->ulid);
        if( $get_card['status'] == false){
            return $this->apiErrorHandle($get_card["message"]);
        }

        $card_api = $get_card['data']['virtual_card'];

        $myCard = [
            "id"            => $myCard->id,
            "reference_id"  => $myCard->reference_id,
            "ulid"          => $myCard->ulid,
            "customer_ulid" => $myCard->customer_ulid,
            "card_name"     => $myCard->card_name,
            "amount"        => $myCard->amount,
            "currency"      => $myCard->currency,
            "card_tier"     => $myCard->card_tier,
            "card_type"     => $myCard->card_type,
            "card_exp_time" => $myCard->card_exp_time,
            "masked_pan"    => $myCard->masked_pan,
            "real_pan"      => $card_api['real_pan'] ?? $myCard->masked_pan,
            "cvv"           => $card_api['cvv'] ?? "***",
            "address"       => $myCard->address,
            "status"        => $myCard->status,
            "env"           => $myCard->env,
            "is_default"    => $myCard->is_default,
            "created_at"    => $myCard->created_at,
            "updated_at"    => $myCard->updated_at,
        ] ;

        $data =[
            'base_curr' => get_default_currency_code(),
            'myCard'=> (object)$myCard,
        ];

        return Response::success([__('card Details')],$data);
    }


    public function freezeUnfreeze(Request $request){

        $validator = Validator::make($request->all(), [
            'status'        => 'required|string',
            'card_id'       => 'required|string',
        ]);

        if ($validator->fails()) {
            return Response::validation($validator->errors()->all(),null);
        }

        $validated = $validator->validate();
        $card      = CardyfieVirtualCard::where('ulid',$validated['card_id'])->first();

        if(!$card){
            return Response::error([__("Something is wrong in your card")],null);
        }

        $get_card  = (new CardyFieHelper())->getCard( (string)$card->ulid);
        if( $get_card['status'] == false){
            return $this->apiErrorHandle($get_card["message"]);
        }

        if($validated['status'] == GlobalConst::CARD_ENABLED_STATUS){
            $make_unfreeze  = (new CardyFieHelper())->unfreeze( (string)$card->ulid);
            if($make_unfreeze['status'] == false){
                return $this->apiErrorHandle($make_unfreeze["message"], true);
            }
            $card->status =  GlobalConst::CARD_ENABLED_STATUS;
            $card->save();

            return Response::success([__('Card unfreeze successfully!')],null);

        }elseif($validated['status'] == GlobalConst::CARD_FREEZE_STATUS){
            $freeze  = (new CardyFieHelper())->freeze( (string)$card->ulid);
            if($freeze['status'] == false){
                return $this->apiErrorHandle($freeze["message"], true);
            }

            $card->status =  GlobalConst::CARD_FREEZE_STATUS;
            $card->save();

            return Response::success([__('Card freeze successfully!')],null);
        }
    }


    public function cardTransaction(){
        $validator = Validator::make(request()->all(), [
            'card_id'     => "required|string",
        ]);
        if($validator->fails()){
            return Response::validation($validator->errors()->all(),null);
        }

        $validated  = $validator->validate();
        $user = auth()->user();
        $card = CardyfieVirtualCard::where('user_id',$user->id)->where('ulid',$validated['card_id'])->first();
        if(!$card){
            return Response::error([__("Something is wrong in your card")],null);
        }

        $get_card_transactions  = (new CardyFieHelper())->getCardTransaction( (string) $card->ulid);

        if( $get_card_transactions['status'] == false){
            return $this->apiErrorHandle($get_card_transactions["message"]);
        }
        $get_card_transactions =  $get_card_transactions['data']['transactions'] ?? [];

        $data =[
            'base_curr' => get_default_currency_code(),
            'card_transactions'=> $get_card_transactions ?? [],
        ];

        return Response::success([__('Virtual Card Transaction')],$data);

    }

    public function closeCard(Request $request) {

        $validator = Validator::make(request()->all(), [
            'card_id'     => "required|string",
        ]);
        if($validator->fails()){
            return Response::validation($validator->errors()->all(),null);
        }

        $validated  = $validator->validate();
        $user       = auth()->user();
        $targetCard = CardyfieVirtualCard::where('ulid',$validated['card_id'])->where('user_id',$user->id)->first();

        if(!$targetCard){
            return Response::error([__("Something is wrong in your card")],null);
        }

        $make_close  = (new CardyFieHelper())->closeCard( (string)$targetCard->ulid);
        if($make_close['status'] == false){
            return $this->apiErrorHandle($make_close["message"]);
        }

        try{
            $targetCard->update([
                'status'         => GlobalConst::CARD_CLOSED_STATUS,
            ]);

        }catch(Exception $e) {
            return Response::error([__("Something went wrong! Please try again.")],null);
        }

        return Response::success([__('Card closed successfully')],null);
    }


    //*****************************Transaction Create Function******************************//


    //for card issues
    public function insertCardBuy($trx_id,$user,$wallet,$card,$charges) {
        $trx_id = $trx_id;
        $authWallet = $wallet;
        $afterCharge = ($authWallet->balance - $charges['payable']);
        $details =[
            'card_info' => $card ?? '',
            'charges'   => $charges,
        ];
        DB::beginTransaction();
        try{
            $id = DB::table("transactions")->insertGetId([
                'user_id'                     => $user->id,
                'user_wallet_id'              => $authWallet->id,
                'payment_gateway_currency_id' => null,
                'type'                        => PaymentGatewayConst::VIRTUALCARD,
                'trx_id'                      => $trx_id,
                'amount'                      => $charges['card_amount'],
                'fixed_charge'                => $charges['fixed_charge'],
                'total_charge'                => $charges['total_charge'],
                'total_payable'               => $charges['payable'],
                'available_balance'           => $afterCharge,
                'remark'                      => PaymentGatewayConst::CARDBUY,
                'details'                     => json_encode($details),
                'attribute'                   => PaymentGatewayConst::RECEIVED,
                'status'                      => GlobalConst::STATUS_CONFIRM_PAYMENT,
                'created_at'                  => now(),
            ]);
            $this->updateSenderWalletBalance($authWallet,$afterCharge);

            DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
        return $id;
    }

    // for deposit
    public function insertCardFund($trx_id,$user,$wallet,$amount,$myCard,$charges) {
        $trx_id = $trx_id;
        $authWallet = $wallet;
        $afterCharge = ($authWallet->balance - $charges['payable']);
        $details =[
            'card_info' =>   $myCard ?? '',
            'charges'   =>   $charges,
        ];
        DB::beginTransaction();
        try{
            $id = DB::table("transactions")->insertGetId([
                'user_id'                     => $user->id,
                'user_wallet_id'              => $authWallet->id,
                'payment_gateway_currency_id' => null,
                'type'                        => PaymentGatewayConst::VIRTUALCARD,
                'trx_id'                      => $trx_id,
                'amount'                      => $amount,
                'fixed_charge'                => $charges['fixed_charge'],
                'total_charge'                => $charges['total_charge'],
                'total_payable'               => $charges['payable'],
                'available_balance'           => $afterCharge,
                'remark'                      => ucwords(PaymentGatewayConst::CARDFUND),
                'details'                     => json_encode($details),
                'attribute'                   => PaymentGatewayConst::RECEIVED,
                'status'                      => true,
                'created_at'                  => now(),
            ]);
            $this->updateSenderWalletBalance($authWallet,$afterCharge);

            DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
        return $id;
    }

    //for withdraw
    public function insertCardWithdrawal($trx_id,$user,$wallet,$amount,$myCard,$charges) {
        $trx_id = $trx_id;
        $authWallet = $wallet;
        $afterCharge = ($authWallet->balance + $charges['payable']);
        $details =[
            'card_info' =>   $myCard ?? '',
            'charges'   =>   $charges,
        ];
        DB::beginTransaction();
        try{
            $id = DB::table("transactions")->insertGetId([
                'user_id'                     => $user->id,
                'user_wallet_id'              => $authWallet->id,
                'payment_gateway_currency_id' => null,
                'type'                        => PaymentGatewayConst::VIRTUALCARD,
                'trx_id'                      => $trx_id,
                'amount'                      => $amount,
                'fixed_charge'                => $charges['fixed_charge'],
                'total_charge'                => $charges['total_charge'],
                'total_payable'               => $charges['payable'],
                'available_balance'           => $afterCharge,
                'remark'                      => ucwords(PaymentGatewayConst::CARDWITHDRAW),
                'details'                     => json_encode($details),
                'attribute'                   => PaymentGatewayConst::RECEIVED,
                'status'                      => true,
                'created_at'                  => now(),
            ]);
            $this->updateSenderWalletBalance($authWallet,$afterCharge);

            DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
        return $id;
    }

    //*****************************Notification Create Function******************************//

    public function insertBuyCardCharge($charges,$user,$id,$card_number) {
        DB::beginTransaction();
        try{
            //notification
            $notification_content = [
                'title'         =>__('Buy Card'),
                'message'       => __('Buy card successful')." ".$card_number??"---- ---- ---- ----",
                'image'         => files_asset_path('user-profile'),
            ];

            UserNotification::create([
                'type'      => NotificationConst::CARD_BUY,
                'user_id'   => $user->id,
                'message'   => $notification_content,
            ]);

            DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
    }

    public function insertFundCardCharge($charges,$user,$id,$card_number,$amount) {
        DB::beginTransaction();
        try{

            //notification
            $notification_content = [
                'title'         =>__("Deposit Amount"),
                'message'       => __("Card fund successful card")." : ".$card_number.' '.get_amount($amount,$charges['card_currency'],2),
                'image'         => files_asset_path('profile-default'),
            ];

            UserNotification::create([
                'type'      => NotificationConst::CARD_FUND,
                'user_id'  => $user->id,
                'message'   => $notification_content,
            ]);

           DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
    }

    public function insertWithdrawalCardCharge($charges,$user,$id,$card_number,$amount) {
        DB::beginTransaction();
        try{
            //notification
            $notification_content = [
                'title'         =>__("Withdraw Amount"),
                'message'       => __("Card Withdrawal Successful Card")." : ".$card_number.' '.get_amount($amount,$charges['card_currency'],2),
                'image'         => files_asset_path('profile-default'),
            ];

            UserNotification::create([
                'type'    => NotificationConst::CARD_WITHDRAW,
                'user_id' => $user->id,
                'message' => $notification_content,
            ]);

           DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
    }




    //*****************************Card Charge Calculation******************************//

     //charge calculation functions
    public function cardIssuesCharge($card_tier,$charges,$wallet,$card_currency){
        if($card_tier == GlobalConst::UNIVERSAL_TIER){
            $card_issues_fixed_fee = $charges->cardyfie_universal_card_issues_fee;
        }else{
            $card_issues_fixed_fee = $charges->cardyfie_platinum_card_issues_fee;
        }
        $sPrecision = get_wallet_precision($wallet->currency);
        $exchange_rate = $wallet->currency->rate/$card_currency->rate;

        $data['exchange_rate']         = $exchange_rate;
        $data['card_amount']           = floatval(0);
        $data['card_currency']         = $card_currency->currency_code;
        $data['card_currency_rate']    = $card_currency->rate;

        $data['from_amount']           = $data['card_amount'] * $exchange_rate;
        $data['from_currency']         = $wallet->currency->code;
        $data['from_currency_rate']    = $wallet->currency->rate;

        $data['percent_charge']        = floatval(0);
        $data['fixed_charge']          = ($card_issues_fixed_fee * $exchange_rate);
        $data['total_charge']          = $data['percent_charge'] + $data['fixed_charge'];
        $data['from_wallet_balance']   = $wallet->balance;
        $data['payable']               = $data['from_amount'] + $data['total_charge'];
        $data['card_platform']         = "CardyFie";
        $data['precision_digit']       = $sPrecision;

        return $data;
    }

    public function cardDepositCharges($amount,$charges,$wallet,$card_currency){
        $sPrecision = get_wallet_precision($wallet->currency);
        $exchange_rate = $wallet->currency->rate/$card_currency->rate;

        $data['exchange_rate']         = $exchange_rate;
        $data['card_amount']           = $amount;
        $data['card_currency']         = $card_currency->currency_code;
        $data['card_currency_rate']    = $card_currency->rate;

        $data['from_amount']           = $amount * $exchange_rate;
        $data['from_currency']         = $wallet->currency->code;
        $data['from_currency_rate']    = $wallet->currency->rate;

        $data['percent_charge']        = 0;
        $data['fixed_charge']          = ($charges->fees->cardyfie_card_deposit_fixed_fee * $exchange_rate);
        $data['total_charge']          = $data['percent_charge'] + $data['fixed_charge'];
        $data['from_wallet_balance']   = $wallet->balance;
        $data['payable']               = $data['from_amount'] + $data['total_charge'];
        $data['card_platform']         = "CardyFie";
        $data['precision_digit']       = $sPrecision;

        return $data;

    }

    public function cardWithdrawCharges($amount,$charges,$wallet,$card_currency){
        $sPrecision = get_wallet_precision($wallet->currency);
        $exchange_rate = $wallet->currency->rate/$card_currency->rate;

        $data['exchange_rate']         = $exchange_rate;
        $data['card_amount']           = $amount;
        $data['card_currency']         = $card_currency->currency_code;
        $data['card_currency_rate']    = $card_currency->rate;

        $data['from_amount']           = $amount * $exchange_rate;
        $data['from_currency']         = $wallet->currency->code;
        $data['from_currency_rate']    = $wallet->currency->rate;

        $data['percent_charge']        = 0;
        $data['fixed_charge']          = ($charges->fees->cardyfie_card_withdraw_fixed_fee * $exchange_rate);
        $data['total_charge']          = $data['percent_charge'] + $data['fixed_charge'];
        $data['from_wallet_balance']   = $wallet->balance;
        $data['payable']               = $data['from_amount'] - $data['total_charge'];
        $data['card_platform']         = "CardyFie";
        $data['precision_digit']       = $sPrecision;

        return $data;

    }


    public function updateSenderWalletBalance($authWallet,$afterCharge) {
        $authWallet->update([
            'balance'   => $afterCharge,
        ]);
    }




    public function adminNotificationFund($trx_id,$charges,$amount,$user,$myCard){
        $notification_content = [
            //email notification
            'subject' => __("Virtual Card (Fund Amount)"),
            'greeting' => __("Virtual Card Information"),
            'email_content' =>__("TRX ID")." : ".$trx_id."<br>".__("Request Amount")." : ".get_amount($amount,$charges['card_currency'],$charges['precision_digit'])."<br>".__("Fees & Charges")." : ".get_amount($charges['total_charge'],$charges['from_currency'],$charges['precision_digit'])."<br>".__("Total Payable Amount")." : ".get_amount($charges['payable'],$charges['from_currency'],$charges['precision_digit'])."<br>".__("Card Masked")." : ".$myCard->masked_pan??"---- ----- ---- ----"."<br>".__("Status")." : ".__("success"),

            //push notification
            'push_title' => __("Virtual Card (Fund Amount)"),
            'push_content' => __('TRX ID')." : ".$trx_id." ".__("Request Amount")." : ".get_amount($amount,$charges['card_currency'],$charges['precision_digit'])." ".__("Card Masked")." : ".$myCard->masked_pan??"---- ----- ---- ----",

            //admin db notification
            'notification_type' =>  NotificationConst::CARD_FUND,
            'admin_db_title' => "Virtual Card Funded",
            'admin_db_message' => "Transaction ID"." : ".$trx_id.",".__("Request Amount")." : ".get_amount($amount,$charges['card_currency'],$charges['precision_digit']).","."Card Masked"." : ".$myCard->masked_pan ?? "---- ----- ---- ----"." (".$user->email.")",
        ];

        try{
            //notification
            (new NotificationHelper())->admin(['admin.virtual.card.logs'])
                                    ->adminDbContent([
                                        'type' => $notification_content['notification_type'],
                                        'title' => $notification_content['admin_db_title'],
                                        'message'  => $notification_content['admin_db_message'],
                                    ])
                                    ->send();


        }catch(Exception $e) {}

    }


    public function adminNotification($trx_id,$charges,$user,$v_card){
        $notification_content = [
            //email notification
            'subject' => __("Virtual Card (Buy Card)"),
            'greeting' => __("Virtual Card Information"),
            'email_content' =>__("TRX ID")." : ".$trx_id."<br>".__("Request Amount")." : ".get_amount($v_card->amount,$charges['card_currency'],$charges['precision_digit'])."<br>".__("Fees & Charges")." : ".get_amount($charges['total_charge'],$charges['from_currency'],$charges['precision_digit'])."<br>".__("Total Payable Amount")." : ".get_amount($charges['payable'],$charges['from_currency'],$charges['precision_digit'])."<br>".__("Card Masked")." : ".$v_card->masked_pan ?? "---- ----- ---- ----"."<br>".__("Status")." : ".__("success"),

            //push notification
            'push_title' => __("Virtual Card (Buy Card)"),
            'push_content' => __('TRX ID')." : ".$trx_id." ".__("Request Amount")." : ".get_amount($v_card->amount,$charges['card_currency'],$charges['precision_digit'])." ".__("Card Masked")." : ".$v_card->masked_pan ??"---- ----- ---- ----",

            //admin db notification
            'notification_type' =>  NotificationConst::CARD_BUY,
            'admin_db_title' => "Virtual Card Buy",
            'admin_db_message' => "Transaction ID"." : ".$trx_id.",".__("Request Amount")." : ".get_amount($v_card->amount,$charges['card_currency'],$charges['precision_digit']).","."Card Masked"." : ".$v_card->masked_pan ?? "---- ----- ---- ----"." (".$user->email.")",
        ];

        try{
            //notification
            (new NotificationHelper())->admin(['admin.virtual.card.logs'])
                                    ->adminDbContent([
                                        'type' => $notification_content['notification_type'],
                                        'title' => $notification_content['admin_db_title'],
                                        'message'  => $notification_content['admin_db_message'],
                                    ])
                                    ->send();


        }catch(Exception $e) {}

    }


    public function adminNotificationWithdraw($trx_id,$charges,$amount,$user,$myCard){
        $notification_content = [
            //email notification
            'subject' => __("Virtual Card (Withdraw Amount)"),
            'greeting' => __("Virtual Card Information"),
            'email_content' =>__("TRX ID")." : ".$trx_id."<br>".__("Request Amount")." : ".get_amount($amount,$charges['card_currency'],$charges['precision_digit'])."<br>".__("Fees & Charges")." : ".get_amount($charges['total_charge'],$charges['from_currency'],$charges['precision_digit'])."<br>".__("Will Receive")." : ".get_amount($charges['payable'],$charges['from_currency'],$charges['precision_digit'])."<br>".__("Card Masked")." : ".$myCard->masked_pan??"---- ----- ---- ----"."<br>".__("Status")." : ".__("success"),

            //push notification
            'push_title' => __("Virtual Card (Withdraw Amount)"),
            'push_content' => __('TRX ID')." : ".$trx_id." ".__("Request Amount")." : ".get_amount($amount,$charges['card_currency'],$charges['precision_digit'])." ".__("Card Masked")." : ".$myCard->masked_pan??"---- ----- ---- ----",

            //admin db notification
            'notification_type' =>  NotificationConst::CARD_WITHDRAW,
            'admin_db_title' => "Virtual Card Withdrawal",
            'admin_db_message' => "Transaction ID"." : ".$trx_id.",".__("Request Amount")." : ".get_amount($amount,$charges['card_currency'],$charges['precision_digit']).","."Card Masked"." : ".$myCard->masked_pan ?? "---- ----- ---- ----"." (".$user->email.")",
        ];

        try{
            //notification
            (new NotificationHelper())->admin(['admin.virtual.card.logs'])
                                    ->adminDbContent([
                                        'type' => $notification_content['notification_type'],
                                        'title' => $notification_content['admin_db_title'],
                                        'message'  => $notification_content['admin_db_message'],
                                    ])
                                    ->send();


        }catch(Exception $e) {}

    }


     public function apiErrorHandle($apiErrors){
        $error = ['error' => []];
        if (isset($apiErrors)) {
            if (is_array($apiErrors)) {
                foreach ($apiErrors as $field => $messages) {
                    if (is_array($messages)) {
                        foreach ($messages as $message) {
                            $error['error'][] = $message;
                        }
                    } else {
                        $error['error'][] = $messages;
                    }
                }
            } else {
                $error['error'][] = $apiErrors;
            }
        }

        $errorMessages = array_map(function($message) {
            return rtrim($message, '.');
        }, $error['error']);

        $errorString = implode(', ', $errorMessages);
        $errorString .= '.';

        return Response::error([$errorString ?? __("Something went wrong! Please try again.")]);

    }
}
